//
// ARF - Augmented Reality Framework (ETSI ISG ARF)
//
// Copyright 2024 ETSI
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Last change: June 2024
//

using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using UnityEngine;

using ETSI.ARF.OpenAPI;
using ETSI.ARF.OpenAPI.WorldStorage;

namespace ETSI.ARF.WorldStorage.REST
{
    public class TrackableRequest : RequestBase<Trackable>
    {
        //
        // Wrapper for the endpoints
        //
        static public Trackable GetTrackableSync(WorldStorageServer ws, Guid UUID)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Request Trackable {UUID}...");
            return apiClient.GetTrackableById(token, UUID);
        }
        
        static public ResponseObject<Trackable> GetTrackableAsync(WorldStorageServer ws, Guid UUID, Action<ResponseObject<Trackable>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Request Trackable {UUID}...");
            ResponseObject<Trackable> ro = new ResponseObject<Trackable>("Request Trackable " + UUID.ToString(), func);
            apiClient.GetTrackableByIdAsync(token, UUID, ro.cancellationToken).ContinueWith(OnReceiveObject<Trackable>, ro);
            return ro;
        }

        static public List<Trackable> GetTrackablesSync(WorldStorageServer ws)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log("[REST] Request Trackables...");
            return apiClient.GetTrackables(token);
        }

        static public ResponseObject<List<Trackable>> GetTrackablesAsync(WorldStorageServer ws, Action<ResponseObject<List<Trackable>>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log("[REST] Request Trackables...");
            ResponseObject<List<Trackable>> ro = new ResponseObject<List<Trackable>>("Request Trackables", func);
            apiClient.GetTrackablesAsync(token, ro.cancellationToken).ContinueWith(OnReceiveListOfObjects<Trackable>, ro);
            return ro;
        }

        static public string CreateTrackableSync(WorldStorageServer ws, Trackable trackable)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            // Add some management stuffs
            if (trackable.UUID == Guid.Empty) trackable.UUID = Guid.NewGuid();
            if (trackable.CreatorUUID == Guid.Empty) trackable.CreatorUUID = System.Guid.Parse("8fb169e2-8910-4cd5-a8f9-b7abff38d013");

            Debug.Log($"[REST] Create Trackable {trackable.UUID}...");
            return apiClient.AddTrackable(token, trackable);
        }

        static public ResponseObject<string> CreateTrackableAsync(WorldStorageServer ws, Trackable trackable, Action<ResponseObject<string>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            // Add some management stuffs
            if (trackable.UUID == Guid.Empty) trackable.UUID = Guid.NewGuid();
            if (trackable.CreatorUUID == Guid.Empty) trackable.CreatorUUID = System.Guid.Parse("8fb169e2-8910-4cd5-a8f9-b7abff38d013");

            Debug.Log($"[REST] Create Trackable {trackable.UUID}...");

            ResponseObject<string> ro = new ResponseObject<string>("Create Trackable " + trackable.Name + " (no UUID)", func);
            apiClient.AddTrackableAsync(token, trackable, ro.cancellationToken).ContinueWith(OnReceiveObject<string>, ro);
            return ro;
        }

        static public string UpdateTrackableSync(WorldStorageServer ws, Trackable trackable)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Update Trackable {trackable.UUID}...");
            return apiClient.ModifyTrackable(token, trackable);
        }

        static public ResponseObject<string> UpdateTrackableAsync(WorldStorageServer ws, Trackable trackable, Action<ResponseObject<string>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Update Trackable {trackable.UUID}...");
            ResponseObject<string> ro = new ResponseObject<string>("Update Trackable " + trackable.UUID.ToString(), func);
            apiClient.ModifyTrackableAsync(token, trackable,ro.cancellationToken).ContinueWith(OnReceiveObject<string>, ro);
            return ro;
        }

        static public string DeleteTrackableSync(WorldStorageServer ws, Guid UUID)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Delete Trackable {UUID}...");
            return apiClient.DeleteTrackable(token, UUID);
        }

        static public ResponseObject<string> DeleteTrackableAsync(WorldStorageServer ws, Guid UUID, Action<ResponseObject<string>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Delete Trackable {UUID}...");
            ResponseObject<string> ro = new ResponseObject<string>("Delete Trackable " + UUID.ToString(), func);
            apiClient.DeleteTrackableAsync(token, UUID, ro.cancellationToken).ContinueWith(OnReceiveObject<string>, ro);
            return ro;
        }
    }
}