﻿#define SIMULATE_ANALYSIS
//#define ALWAYS_CALL_REST_API

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using ETSI.ARF.OpenAPI.WorldStorage;
using ETSI.ARF.OpenAPI.WorldAnalysis;

using ETSI.ARF.WorldStorage;
using ETSI.ARF.WorldStorage.REST;
using Newtonsoft.Json;
using System.Numerics;


namespace WorldAnalysisWrapper
{
    class WorldAnalysis
    {
        static public List<Trackable> trackables;
        static public List<WorldAnchor> anchors;
        static public List<WorldLink> links;

        float randomScale = 0.012f;  // m

        public class SubscriptionInfo
        {
            public Guid uuidSub;           // id of subscription (id is defined by the WA server)

            public SubscriptionSingleRequest subscription;
            public Mode_WorldAnalysis mode;

            // Monitor
            public int sendCnt;

            // Results
            public Pose pose;
            public Trackable trackable;
            public ETSI.ARF.OpenAPI.WorldStorage.Transform3D transform3d;

            public Trackable extra;
        }

        private int simulationDelay = 500; // ms
        private string modulename = "";
        private SubscriptionInfo lastSubscription;
        private Dictionary<Guid, SubscriptionInfo> subscriptions = new Dictionary<Guid, SubscriptionInfo>();

        // WS
        private ETSI.ARF.OpenAPI.WorldStorage.RelocalizationInformations relocInfos;

        private bool sendPose = false;

        // Credencials
        private WorldStorageServer server;
        private WorldStorageUser user;
        private bool isRegistered = false;

        private WebSocketSharp.WebSocket webSocket;
        private bool websocketConnected = false;
        private string webserver_url;
        private string websocket_url;

        // Capabilities of the module
        private ETSI.ARF.OpenAPI.WorldStorage.EncodingInformationStructure enc = null;
        private List<ETSI.ARF.OpenAPI.WorldStorage.Capability> capabilities = new List<ETSI.ARF.OpenAPI.WorldStorage.Capability>();


        public WorldAnalysis(string webserver, string websocket, int mode)
        {
            webserver_url = webserver;
            websocket_url = websocket;
            ETSI.ARF.OpenAPI.WorldStorage.Capability cap = null;

            // Simulate a World Analysis module
            if (mode == 1)
            {
                modulename = "HHI-ImageTracker";

                enc = new ETSI.ARF.OpenAPI.WorldStorage.EncodingInformationStructure()
                {
                    Version = "HHI-0.1.0",        // version
                    DataFormat = ETSI.ARF.OpenAPI.WorldStorage.EncodingInformationStructureDataFormat.OTHER       // type/company?
                };

                // Capability #1
                cap = new ETSI.ARF.OpenAPI.WorldStorage.Capability();
                cap.TrackableType = ETSI.ARF.OpenAPI.WorldStorage.TrackableType.IMAGE_MARKER;
                cap.EncodingInformation = enc;
                cap.Framerate = 30;
                cap.Latency = 0.05;
                cap.Accuracy = 01;
                capabilities.Add(cap);
            }
            else if (mode == 2)
            {
                modulename = "HHI-MapAndMesh-Detection";

                enc = new ETSI.ARF.OpenAPI.WorldStorage.EncodingInformationStructure()
                {
                    Version = "HHI-0.1.2",        // version
                    DataFormat = ETSI.ARF.OpenAPI.WorldStorage.EncodingInformationStructureDataFormat.OTHER       // type/company?
                };

                // Capability #1
                cap = new ETSI.ARF.OpenAPI.WorldStorage.Capability();
                cap.TrackableType = ETSI.ARF.OpenAPI.WorldStorage.TrackableType.MAP;
                cap.EncodingInformation = enc;
                cap.Framerate = 6;
                cap.Latency = 0.5;
                cap.Accuracy = 0.8;
                capabilities.Add(cap);

                // Capability #2
                cap = new ETSI.ARF.OpenAPI.WorldStorage.Capability();
                cap.TrackableType = ETSI.ARF.OpenAPI.WorldStorage.TrackableType.MESH;
                cap.EncodingInformation = enc;
                cap.Framerate = 8;
                cap.Latency = 0.5;
                cap.Accuracy = 0.9;
                capabilities.Add(cap);
            }

            //
            // REST
            //

            // Test the REST World Storage API
            server = new WorldStorageServer();
            server.serverName = "HHI";
            server.company = "Fraunhofer HHI";
            server.basePath = webserver_url;
            server.port = 0;

            user = new WorldStorageUser();
            user.userName = "HHI";
            user.company = "Fraunhofer HHI";
            user.UUID = "b4380ebb-eb56-41c2-aeaa-146315a39606";

            // Some admin stuffs
            string res = AdminRequest.PingSync(server);
            Console.WriteLine("Sending 'ping', got response: " + res);

            res = AdminRequest.AdminSync(server);
            Console.WriteLine("Sending 'admin', got response: " + res);

            res = AdminRequest.VersionSync(server);
            Console.WriteLine("Sending 'version', got response: " + res);

            //
            // Reloc test
            //
            //RelocTest();
        }

        ~WorldAnalysis()
        {
            WebSocketClient_Close();
        }


        public bool isRunning()
        {
            return websocketConnected && webSocket.ReadyState == WebSocketSharp.WebSocketState.Open;
        }

        public void Start()
        {
            //
            // Get all lists from the World Storage REST server (test)
            //
#if ALWAYS_CALL_REST_API
            if (Program.isDev)
#endif
            {
                // Get trackables
                trackables = TrackableRequest.GetTrackablesSync(server);
                Console.WriteLine("Querying Trackables: got list with " + trackables.Count + " items: ");
                foreach (var t in trackables)
                {
                    Console.WriteLine($"\tUUID: { t.UUID }  Name: { t.Name } (Type: { t.TrackableType })");
                }

                // Get anchors
                anchors = WorldAnchorRequest.GetWorldAnchorsSync(server);
                Console.WriteLine("Querying World Anchors: got list with " + anchors.Count + " items: ");
                foreach (var a in anchors)
                {
                    Console.WriteLine($"\tUUID: { a.UUID }  Name: { a.Name }");
                }

                // Get links
                links = WorldLinkRequest.GetWorldLinksSync(server);
                Console.WriteLine("Querying World Links: got list with " + links.Count + " items: ");
                foreach (var l in links)
                {
                    //Console.WriteLine($"\tUUID: { l.UUID }  From: { l.UUIDFrom } To: { l.UUIDTo }");
                    Console.WriteLine($"\tUUID: { l.UUID }");
                }
            }

            //
            // Start the WebSockets (World Analysis server)
            //
            webSocket = new WebSocketSharp.WebSocket(websocket_url);

            Console.WriteLine($"[WS] Try connecting the WA server as { modulename}...");

            // Define standard callbacks
            webSocket.OnOpen += (sender, e) =>
            {
                Console.WriteLine("[WS] Connected");
                websocketConnected = true;

                Console.WriteLine($"[WS] Send registration request to server for {modulename}");
                webSocket.Send($"RegisterModule={modulename}");
            };
            webSocket.OnClose += (sender, e) =>
            {
                Console.WriteLine("[WS] Disconnected");
                websocketConnected = false;
            };
            webSocket.OnMessage += (sender, e) => WebSocketClient_OnReceive(e.Data);  // main event
            webSocket.OnError += (sender, e) => Console.WriteLine("[WS] Websocket error!");
            webSocket.Connect();
        }

        public string ToJson(object o)
        {
            return JsonConvert.SerializeObject(o, Formatting.Indented);
        }

        private void WebSocketClient_Close()
        {
            if (websocketConnected)
            {
                webSocket.Send("UnregisterModule=" + modulename);
                webSocket.Close();
                webSocket = null;
            }
        }

        public void WebSocketClient_Send(string msg)
        {
            Console.WriteLine($"[WS] Send to WA server: {msg}");
            webSocket?.Send(msg);
        }

        public void _webSocketClient_SendPose(SubscriptionInfo si)
        {
            //Console.WriteLine($"[WS] #{sendCnt } Send to WA server: {msg}");
            si.sendCnt++;
            Console.WriteLine($"[WS] Send pose #{ si.sendCnt } to WA server (ID: { si.uuidSub })");
            webSocket?.Send("NewPose=" + si.pose.ToJson());
        }

        private void WebSocketClient_SendCapabilities()
        {
            foreach (var cap in capabilities)
            {
                string cap_json = ToJson(cap);
                WebSocketClient_Send("Capabilities=" + modulename + "=" + cap_json);
            }
        }

        public void WebSocketClient_SendPose(SubscriptionInfo info)
        {
            Guid uuidTarget = info.subscription.Target;
            ETSI.ARF.OpenAPI.WorldStorage.Transform3D matrix = null;

            Trackable tr = null;
            WorldAnchor an = null;

#if ALWAYS_CALL_REST_API
            try
            {
                tr = TrackableRequest.GetTrackableSync(server, uuidTarget);
                matrix = tr.LocalCRS;
            }
            catch
            {
                try
                {
                    an = WorldAnchorRequest.GetWorldAnchorSync(server, uuidTarget);
                    matrix = an.LocalCRS;
                }
                catch { }
            }
#else
            foreach (var item in trackables)
            {
                if (item.UUID == uuidTarget)
                {
                    tr = item;
                    matrix = tr.LocalCRS;
                    break;
                }
            }

            foreach (var item in anchors)
            {
                if (item.UUID == uuidTarget)
                {
                    an = item;
                    matrix = an.LocalCRS;
                    break;
                }
            }

#endif
            if (matrix == null)
            {
                WebSocketClient_Send("Error! Pose cannot be calculated (missing trackable or world-anchor)");
                return;
            }

            //
            // Calculate/Get the current pose of the uuid
            //
            Pose pose = new Pose();
            pose.Uuid = uuidTarget;
            pose.Mode = info.mode;
            pose.SubscriptionUrl = websocket_url;

            MatrixPoseValue pv = new MatrixPoseValue();
            pv.Type = PoseValueType.MATRIX;
            pv.Unit = ETSI.ARF.OpenAPI.WorldAnalysis.UnitSystem.M;

            //
            // Get the reloc infos ?
            //
            if (info.trackable == null)
            {
#if !SIMULATE_ANALYSIS
                List<Guid> uuids = new List<Guid> { uuidTarget };
                List<Mode_WorldStorage> modes = new List<Mode_WorldStorage> { Mode_WorldStorage.REQUEST_TO_TRACKABLES };

                RelocalizationInformations relocInfos = RelocalizationInformationRequest.GetRelocalizationInformationSync(server, uuids, modes, capabilities);
                foreach (var i in relocInfos.RelocInfo)
                {
                    if (i.RelocObjects.Count > 0)
                    {
                        RelocObjects ro = i.RelocObjects.ToList<RelocObjects>()[0];
                        info.trackable = ro.Trackable;
                        info.transform3d = ro.Transform3D;
                        break;
                    }
                }
#else
                // workaround - validation/simulation (STF669)
                //if (info.trackable == null)
                {
                    foreach (var item in links)
                    {
                        if (item.UUIDTo == uuidTarget)
                        {
#if ALWAYS_CALL_REST_API
                            try
                            {
                                Trackable t = TrackableRequest.GetTrackableSync(server, item.UUIDFrom);
                                info.trackable = t;
                                info.transform3d = t.LocalCRS;
                            }
                            catch { }
#else
                            foreach (var t in trackables)
                            {
                                if (t.UUID == item.UUIDFrom)
                                {
                                    info.trackable = t;
                                    info.transform3d = t.LocalCRS;
                                    break;
                                }
                            }
#endif
                        }


                        if (item.UUIDFrom == uuidTarget)
                        {
#if ALWAYS_CALL_REST_API
                            try
                            {
                                Trackable t = TrackableRequest.GetTrackableSync(server, item.UUIDTo);
                                info.trackable = t;
                                info.transform3d = t.LocalCRS;
                            }
                            catch { }
#else
                            foreach (var t in trackables)
                            {
                                if (t.UUID == item.UUIDTo)
                                {
                                    info.trackable = t;
                                    info.transform3d = t.LocalCRS;
                                    break;
                                }
                            }
#endif
                        }
                    }
#endif
                }
            }

            if (info.trackable != null)
            {
                pose.InstructionInfo = "Workaround for STF669 (parent; w/no RelocInfo)";

                //
                // Direct parent request (without calling RelocInfo!)
                //
                ETSI.ARF.OpenAPI.WorldStorage.Transform3D parentMatrix = info.transform3d;

                Matrix4x4 _parentMatrix = parentMatrix.Matrix();
                Matrix4x4 _matrix = matrix.Matrix(); // target
                Matrix4x4 m = _matrix * _parentMatrix;  // target * parent

                // todo: Invert?

                // Transpose one matrix to the other
                pv.Transform = new ETSI.ARF.OpenAPI.WorldAnalysis.Transform3D()
                {
                    m.M11, m.M12, m.M13, m.M14,
                    m.M21, m.M22, m.M23, m.M24,
                    m.M31, m.M32, m.M33, m.M34,
                    m.M41, m.M42, m.M43, m.M44,
                };
            }
            else
            {
                pose.InstructionInfo = "Workaround for STF669 (identity pose)";

                //
                // Return identiy
                //

                // Transpose one matrix to the other
                pv.Transform = new ETSI.ARF.OpenAPI.WorldAnalysis.Transform3D()
                {
                    matrix[0], matrix[1], matrix[2], matrix[3],
                    matrix[4], matrix[5], matrix[6], matrix[7],
                    matrix[8], matrix[9], matrix[10], matrix[11],
                    matrix[12], matrix[13], matrix[14], matrix[15]
                };
            }

#if SIMULATE_ANALYSIS
            // Simulate jitter in position
            Random r = new Random();
            pv.Transform[3] = pv.Transform[3] + (float)r.NextDouble() * randomScale;
            pv.Transform[7] = pv.Transform[7] + (float)r.NextDouble() * randomScale;
            pv.Transform[11] = pv.Transform[11] + (float)r.NextDouble() * randomScale;
#endif

            // Set the new pose
            pose.Value = pv;
            info.pose = pose;

            //Console.WriteLine($"[WS] Found pose for uuid={ uuidTarget }...");

            //  Send the pose
            _webSocketClient_SendPose(info);
        }

        // Events
        private void WebSocketClient_OnReceive(string serverMessage)
        {
            if (serverMessage.Contains("You are now registered"))
            {
                isRegistered = true;
                Console.WriteLine($"[WS] {serverMessage }");
                //Console.WriteLine($"[WS] Registration of {modulename} was succesfull.");

                WebSocketClient_SendCapabilities();

                Console.WriteLine($"[WS] Entering the websockets main loop...");
            }
            else if (isRegistered)
            {
                if (serverMessage == "UnregisterModuleOK")
                {
                    // Stop the loop and disconnect the server
                    sendPose = false;
                }
                else if (serverMessage.StartsWith("SubscribeSimulatedPose="))   // simulation for STFT669 workaround / SylR
                {
                    Guid uuidSub = Guid.Parse(serverMessage.Split('=')[1]);
                    string json_subPose = serverMessage.Split('=')[2];
                    string _trackableUUID = serverMessage.Split('=')[3]; // extra field

                    SubscriptionInfo si = new SubscriptionInfo()
                    {
                        uuidSub = uuidSub,
                    };
                    si.subscription = JsonConvert.DeserializeObject<SubscriptionSingleRequest>(json_subPose);
                    si.mode = si.subscription.Mode;
                    Console.WriteLine($"[WS] New pose subscription (simulation) for UUID={ si.subscription.Target }");

                    // receive also the parent object (trackable) from the client?
                    si.extra = TrackableRequest.GetTrackableSync(server, Guid.Parse(_trackableUUID));
                    Console.WriteLine($"Found trackable (UUID={ _trackableUUID }) for simulation: name = { si.extra.Name }");

                    // Update lists
                    if (!subscriptions.ContainsKey(si.subscription.Target)) subscriptions.Add(si.subscription.Target, si);
                    if (subscriptions.Count > 0) sendPose = true;

                    lastSubscription = si;

                    WebSocketClient_Send("PoseIsNowSubscribed");
                }
                else if (serverMessage.StartsWith("SubscribePose="))
                {
                    Guid uuidSub = Guid.Parse(serverMessage.Split('=')[1]);
                    string json_subPose = serverMessage.Split('=')[2];

                    SubscriptionInfo si = new SubscriptionInfo()
                    {
                        uuidSub = uuidSub,
                    };
                    si.subscription = JsonConvert.DeserializeObject<SubscriptionSingleRequest>(json_subPose);
                    si.mode = si.subscription.Mode;
                    Console.WriteLine($"[WS] New pose subscription for UUDI={ si.subscription.Target }");

                    // Update lists
                    if (!subscriptions.ContainsKey(si.subscription.Target)) subscriptions.Add(si.subscription.Target, si);
                    if (subscriptions.Count > 0) sendPose = true;

                    WebSocketClient_Send("PoseIsNowSubscribed");
                }
                else if (serverMessage.StartsWith("UnsubscribePose="))
                {
                    Guid uuidSub = Guid.Parse(serverMessage.Split('=')[1]);

                    // Update lists
                    foreach (var si in subscriptions.Values)
                    {
                        if (si.uuidSub == uuidSub)
                        {
                            // remove
                            subscriptions.Remove(si.subscription.Target);
                        }
                    }
                    if (subscriptions.Count == 0) sendPose = false;

                    WebSocketClient_Send("PoseIsNowUnsubscribed");
                }
                else if (serverMessage.StartsWith("ConfigureFramerate="))
                {
                    // todo: check if the framerate is authorized, otherwise send an error back
                    int fps = int.Parse(serverMessage.Split('=')[1]);

                    // Not implemented
                    WebSocketClient_Send("Error=Server: Setting framerate is not Implemented!");
                }
                //
                // Special msg for simulating a global user's position
                //
                else if (serverMessage.StartsWith("CurrentUserPose="))
                {
                    string _userPose = serverMessage.Split('=')[1];
                    Pose userPose = JsonConvert.DeserializeObject<Pose>(_userPose);
                    Console.WriteLine($"New user current position is: { userPose.ToJson() }");
                }
                //
                // Pose requests
                //                    
                else if (serverMessage.StartsWith("GetPose="))  // 2 args
                {
                    Guid uuidTarget = Guid.Parse(serverMessage.Split('=')[1]);
                    Mode_WorldAnalysis mode = Mode_WorldAnalysis.TRACKABLES_TO_DEVICE; // serverMessage.split(':')[2];

                    SubscriptionSingleRequest ss = new SubscriptionSingleRequest();
                    ss.Target = uuidTarget;
                    SubscriptionInfo si = new SubscriptionInfo()
                    {
                        uuidSub = Guid.Empty,       // without subscription!
                        mode = mode
                    };
                    WebSocketClient_SendPose(si);
                }
                else if (serverMessage == "PoseStop")
                {
                    //SetColor(Color.yellow);
                }
                else if (serverMessage == "RequestNextPose" && sendPose == true)
                {
                    System.Threading.Thread.Sleep(simulationDelay);
                    foreach (var si in subscriptions.Values)
                    {
                        WebSocketClient_SendPose(si);
                    }
                }
            }
        }

        private void RelocTest()
        {
            RelocalizationInformation relocInfo = null;
            Guid tr_uuid = new Guid("6a555bf1-fceb-4a8c-a648-191087ad1b21");

            Guid an_uuid = new Guid("da1284ee-e895-4434-906f-5599b92ab51a");
            List<Guid> uuids = new List<Guid>
            {
               an_uuid
            };

            List<Mode_WorldStorage> modes = new List<Mode_WorldStorage>
            {
                Mode_WorldStorage.TRACKABLES_TO_REQUEST
            };

            /// Collect relocalization information
            relocInfos = RelocalizationInformationRequest.GetRelocalizationInformation(server, uuids, modes, capabilities);
            if (relocInfos == null)
            {
                Console.WriteLine("ETSI ARF GetRelocalizationInformation : request response is null");
            }
            relocInfo = relocInfos.RelocInfo.First(); //Only one uuid requested
            Console.WriteLine($"Found Reloc: num = { relocInfo.RelocObjects.Count }");
        }
    }
}
