//
// ARF - Augmented Reality Framework (ETSI ISG ARF)
//
// Copyright 2024 ETSI
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Last change: June 2024
//

using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Net.WebSockets;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Http;
using Swashbuckle.AspNetCore.Annotations;
using Swashbuckle.AspNetCore.SwaggerGen;
using Newtonsoft.Json;

using ETSI.ARF.OpenAPI.WorldAnalysis.Models;

#pragma warning disable CS1591 // Fehlendes XML-Kommentar fr ffentlich sichtbaren Typ oder Element
namespace ETSI.ARF.OpenAPI.WorldAnalysis.Controllers
{
    //
    // ETSI-ARF World Analysis WebSocket implementation
    // see also: https://learn.microsoft.com/de-de/aspnet/core/fundamentals/websockets?view=aspnetcore-5.0
    //
    public class WebSocketController : ControllerBase
    {
        private string currentName = "";
        private bool registered = false;
        private bool firstTime = true;
        private int timeCnt = 3;

        [HttpGet("/ws")]
        public async Task Get()
        {
            if (HttpContext.WebSockets.IsWebSocketRequest)
            {
                using var webSocket = await HttpContext.WebSockets.AcceptWebSocketAsync();

                if (webSocket.State == WebSocketState.Connecting)
                {
                    // Response an OK message?
                }
                else if (webSocket.State == WebSocketState.Open)
                {
                    await HandleClientData(HttpContext, webSocket);
                }
            }
            else
            {
                await HttpContext.Response.WriteAsync("ETSI-ARF World Analysis: Not a valid WebSocket request.");
                HttpContext.Response.StatusCode = StatusCodes.Status400BadRequest;
            }
        }

        //
        // Send a line of text
        //
        private async Task SendText(WebSocket webSocket, string text)
        {
            // Response an OK message
            var message = text;
            var bytes = Encoding.UTF8.GetBytes(message);
            var arraySegment = new ArraySegment<byte>(bytes, 0, bytes.Length);
            await webSocket.SendAsync(arraySegment, WebSocketMessageType.Text, true, CancellationToken.None);
        }

        //
        // Send the time all seconds
        //
        //[HttpGet("/ws/time")]
        private async Task SendTime(WebSocket webSocket)
        {
            while (true)
            {
                var message = "ARF World Analysis Server, Time = " + DateTime.Now.ToLocalTime();

                if (webSocket.State == WebSocketState.Open)
                {
                    timeCnt--;
                    if (timeCnt < 0)
                    {
                        await SendText(webSocket, "TimeStop");
                        break;
                    }
                    else await SendText(webSocket, message);
                }
                else if (webSocket.State == WebSocketState.Closed || webSocket.State == WebSocketState.Aborted)
                {
                    break;
                }
                Thread.Sleep(1000);
            }
        }

        float rotInc = 0;
        private async Task SendPose(WebSocket webSocket)
        {
            while (true)
            {
                if (webSocket.State == WebSocketState.Open)
                {
                    timeCnt--;
                    if (timeCnt < 0)
                    {
                        await SendText(webSocket, "PoseStop");
                        break;
                    }
                    else
                    {
                        PoseValue val = new PoseValue();
                        val.Unit = UnitSystem.MEnum;
                        val.Position = new List<float>() { 1, 2, 3 };
                        val.Rotation = new List<float>() { rotInc, 0, 0, 0 };
                        rotInc += 0.01f;

                        Pose pose = new Pose();
                        pose.Uuid = new Guid();
                        pose.EstimationState = Pose.EstimationStateEnum.OKEnum;
                        //pose.SubscriptionUrl = "";
                        pose.Value = val;
                        pose.Timestamp = (int)DateTime.Now.ToFileTime();
                        pose.Confidence = 1;
                        pose.Mode = ModeWorldAnalysis.DEVICETOTRACKABLESEnum;

                        string json = pose.ToJson();

                        await SendText(webSocket, json);
                    }
                }
                else if (webSocket.State == WebSocketState.Closed || webSocket.State == WebSocketState.Aborted)
                {
                    if (WorldAnalysisModules.Singleton.modules.Contains(currentModule)) WorldAnalysisModules.Singleton.modules.Remove(currentModule);
                    if (WorldAnalysisModules.Singleton.clients.Contains(currentClient)) WorldAnalysisModules.Singleton.clients.Remove(currentClient);
                    currentClient = null;
                    currentModule = null;
                    break;
                }
                Thread.Sleep(250);
            }
        }

        private Module currentModule;
        private Client currentClient;

        private async void OnReceiveText(WebSocket webSocket, string msg)
        {
            if (firstTime)
            {
                // Register the client/module
                if (msg.StartsWith("Module:"))
                {
                    registered = true;
                    firstTime = false;
                    currentName = msg.Split(':')[1];
                    if (currentModule == null && currentClient == null)
                    {
                        currentModule = new Module();
                        currentModule.name = currentName;
                        WorldAnalysisModules.Singleton.modules.Add(currentModule);
                        await SendText(webSocket, "ARF World Analysis Server: You are now registered as a module: " + currentName);
                    }
                }
                else if (msg.StartsWith("Client:"))
                {
                    registered = true;
                    firstTime = false;
                    currentName = msg.Split(':')[1];
                    if (currentModule == null && currentClient == null)
                    {
                        currentClient = new Client();
                        currentClient.name = currentName;
                        WorldAnalysisModules.Singleton.clients.Add(currentClient);
                        await SendText(webSocket, "ARF World Analysis Server: You are now registered as a client: " + currentName);
                    }
                }
                else
                {
                    registered = false;
                    await SendText(webSocket, "ARF World Analysis Server: Cannot register " + msg);
                }
            }
            else if (registered)
            {
                if (msg.StartsWith("StartSendingPose"))
                {
                    if (msg.Contains(':')) timeCnt = int.Parse(msg.Split(':')[1]);
                    else timeCnt = 3;
                    await SendPose(webSocket);
                }
                else if (msg.StartsWith("Time"))
                {
                    if (msg.Contains(':')) timeCnt = int.Parse(msg.Split(':')[1]);
                    else timeCnt = 3;
                    await SendTime(webSocket);
                }
                else if (msg.StartsWith("Capabilities"))
                {
                    // Client is sending its capabilities
                    string[] str_cap = msg.Split('=');
                    Capability _c = JsonConvert.DeserializeObject<Capability>(str_cap[1]);
                    if (currentModule != null) currentModule.capabilities.Add(_c);
                }
                else if (msg == "UnregisterClient")
                {
                    // Unregister client/user
                    currentName = "";
                    firstTime = true;
                    registered = false;
                    await SendText(webSocket, "SessionStop");
                }
                else if (msg == "Idle")
                {
                    await SendText(webSocket, "Idle");
                }
                else if (msg == "Busy")
                {
                    await SendText(webSocket, "Busy");
                }
                else
                {
                    // Send a response
                    await SendText(webSocket, "ARF World Analysis Server: I got this unknown message: " + msg);
                }
            }
        }

        private async Task HandleClientData(HttpContext context, WebSocket webSocket)
        {
            var buffer = new byte[1024 * 4];

            // Read/get the first data block
            WebSocketReceiveResult result = await webSocket.ReceiveAsync(new ArraySegment<byte>(buffer), CancellationToken.None);
            if (result.MessageType == WebSocketMessageType.Text)
            {
                string getMsg = System.Text.Encoding.UTF8.GetString(buffer, 0, result.Count);
                OnReceiveText(webSocket, getMsg);
            }

            // Entering the loop
            while (!result.CloseStatus.HasValue)
            {
                // Read/get the next block
                result = await webSocket.ReceiveAsync(new ArraySegment<byte>(buffer), CancellationToken.None);
                if (result.MessageType == WebSocketMessageType.Text)
                {
                    string getMsg = System.Text.Encoding.UTF8.GetString(buffer, 0, result.Count);
                    OnReceiveText(webSocket, getMsg);
                }
            }
            await webSocket.CloseAsync(result.CloseStatus.Value, result.CloseStatusDescription, CancellationToken.None);
        }


        //
        // Echo incoming messages
        //
        private async Task Echo(WebSocketReceiveResult context, WebSocket webSocket)
        {
            var buffer = new byte[1024 * 4];

            // Read/get the first data block
            WebSocketReceiveResult result = await webSocket.ReceiveAsync(new ArraySegment<byte>(buffer), CancellationToken.None);

            // Send/echo the message back to the client
            await SendText(webSocket, "ARF World Analysis Server: I got this (raw) message: ");
            await webSocket.SendAsync(new ArraySegment<byte>(buffer, 0, result.Count), result.MessageType, result.EndOfMessage, CancellationToken.None);
        }
    }
}
#pragma warning restore CS1591 // Fehlendes XML-Kommentar fr ffentlich sichtbaren Typ oder Element


