//
// ARF - Augmented Reality Framework (ETSI ISG ARF)
//
// Copyright 2024 ETSI
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Last change: June 2024
//

using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Net.WebSockets;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Http;
using Swashbuckle.AspNetCore.Annotations;
using Swashbuckle.AspNetCore.SwaggerGen;
using Newtonsoft.Json;

using ETSI.ARF.OpenAPI.WorldAnalysis.Models;

#pragma warning disable CS1591 // Fehlendes XML-Kommentar fr ffentlich sichtbaren Typ oder Element
namespace ETSI.ARF.OpenAPI.WorldAnalysis.Controllers
{
    //
    // ETSI-ARF World Analysis WebSocket implementation
    // see also: https://learn.microsoft.com/de-de/aspnet/core/fundamentals/websockets?view=aspnetcore-5.0
    //
    public class WebSocketController : ControllerBase
    {
        private WebSocket websocket;
        static public int WebSocketControllerInstanceCount = 0;

        private string currentName = "";
        private bool registered = false;
        private bool firstTime = true;
        private int timeCnt = 3;

        [HttpGet("/ws")]
        public async Task Get()
        {
            if (HttpContext.WebSockets.IsWebSocketRequest)
            {
                using var webSocket = await HttpContext.WebSockets.AcceptWebSocketAsync();

                if (webSocket.State == WebSocketState.Connecting)
                {
                    // Response an OK message?
                }
                else if (webSocket.State == WebSocketState.Open)
                {
                    // Go to the loop...
                    await WebSocketServer_Loop(HttpContext, webSocket);
                }
            }
            else
            {
                await HttpContext.Response.WriteAsync("ETSI-ARF World Analysis: Not a valid WebSocket request.");
                HttpContext.Response.StatusCode = StatusCodes.Status400BadRequest;
            }
        }

        //
        // Send a line of text
        //
        static private async Task SendText(WebSocket ws, string msg)
        {
            var message = msg;
            var bytes = Encoding.UTF8.GetBytes(message);
            var arraySegment = new ArraySegment<byte>(bytes, 0, bytes.Length);
            await ws.SendAsync(arraySegment, WebSocketMessageType.Text, true, CancellationToken.None);
        }

        public void SendText(string msg)
        {
            var message = msg;
            var bytes = Encoding.UTF8.GetBytes(message);
            var arraySegment = new ArraySegment<byte>(bytes, 0, bytes.Length);
            websocket.SendAsync(arraySegment, WebSocketMessageType.Text, true, CancellationToken.None);
        }

        //
        // Send a demo pose all seconds
        //
        float rotInc = 0;
        private async Task SendDemoPose(WebSocket ws)
        {
            while (true)
            {
                if (ws.State == WebSocketState.Open)
                {
                    timeCnt--;
                    if (timeCnt < 0)
                    {
                        await SendText(ws, "PoseStop");
                        break;
                    }
                    else
                    {
                        PoseValue val = new PoseValue();
                        val.Unit = UnitSystem.MEnum;
                        val.Position = new List<float>() { 1, 2, 3 };
                        val.Rotation = new List<float>() { rotInc, 0, 0, 0 };
                        rotInc += 0.01f;

                        Pose pose = new Pose();
                        pose.Uuid = new Guid();
                        pose.EstimationState = Pose.EstimationStateEnum.OKEnum;
                        //pose.SubscriptionUrl = "";
                        pose.Value = val;
                        pose.Timestamp = (int)DateTime.Now.ToFileTime();
                        pose.Confidence = 1;
                        pose.Mode = ModeWorldAnalysis.DEVICETOTRACKABLESEnum;

                        string json = pose.ToJson();

                        await SendText(ws, "NewPose=" + json);
                    }
                }
                else if (ws.State == WebSocketState.Closed || ws.State == WebSocketState.Aborted)
                {
                    if (WorldAnalysisConnections.Singleton.modules.Contains(currentModule)) WorldAnalysisConnections.Singleton.modules.Remove(currentModule);
                    if (WorldAnalysisConnections.Singleton.clients.Contains(currentClient)) WorldAnalysisConnections.Singleton.clients.Remove(currentClient);
                    currentClient = null;
                    currentModule = null;
                    break;
                }
                Thread.Sleep(250);
            }
        }

        private Module currentModule;
        private Client currentClient;

        private async void OnReceiveText(string msg)
        {
            #region Register the client/module
            if (firstTime)
            {
                if (msg.StartsWith("RegisterModule="))
                {
                    registered = true;
                    firstTime = false;
                    currentName = msg.Split('=')[1];

                    // If module exist already reuse it
                    Module module = null;
                    foreach (var item in WorldAnalysisConnections.Singleton.modules) if (item.name == currentName) module = item;
                    if (module != null)
                    {
                        currentModule = module;
                        currentModule.name = currentName;
                        currentModule.websockets = this;
                        currentModule.capabilities.Clear();
                    }

                    if (currentModule == null && currentClient == null)
                    {
                        currentModule = new Module();
                        currentModule.name = currentName;
                        currentModule.websockets = this;
                        WorldAnalysisConnections.Singleton.modules.Add(currentModule);
                    }
                    SendText($"ARF World Analysis Server: #{ WebSocketControllerInstanceCount } You are now registered as module: { currentName }");
                }
                else if (msg.StartsWith("RegisterClient="))
                {
                    registered = true;
                    firstTime = false;
                    currentName = msg.Split('=')[1];
                    if (currentModule == null && currentClient == null)
                    {
                        currentClient = new Client();
                        currentClient.name = currentName;
                        currentClient.websockets = this;
                        WorldAnalysisConnections.Singleton.clients.Add(currentClient);
                        SendText($"ARF World Analysis Server: #{ WebSocketControllerInstanceCount } You are now registered as client: { currentName }");
                    }
                }
                else
                {
                    registered = false;
                    SendText("ARF World Analysis Server: Cannot register " + msg);
                }
                return;
            }
            #endregion

            if (registered)
            {
                //
                // Some admin stuffs
                //
                if (msg == "Idle")
                {
                    SendText("Idle");
                }
                else if (msg == "Busy")
                {
                    SendText("Busy");
                }
                else if (msg.StartsWith("UnregisterModule="))
                {
                    string name = msg.Split('=')[1];

                    // Unregister a client (e.g. Unity client)
                    Module module = null;
                    foreach (var item in WorldAnalysisConnections.Singleton.modules) if (item.name == name) module = item;
                    if (module != null)
                    {
                        WorldAnalysisConnections.Singleton.modules.Remove(module);
                        SendText("UnregisterModuleOK");
                    }
                    currentName = "";
                    firstTime = true;
                    registered = false;
                }
                else if (msg.StartsWith("UnregisterClient="))
                {
                    string name = msg.Split('=')[1];

                    // Unregister a client (e.g. Unity client)
                    Client client = null;
                    foreach (var item in WorldAnalysisConnections.Singleton.clients) if (item.name == name) client = item;
                    if (client != null)
                    {
                        WorldAnalysisConnections.Singleton.clients.Remove(client);
                        SendText("UnregisterClientOK");
                    }
                    currentName = "";
                    firstTime = true;
                    registered = false;
                }
                //
                // OpenAPI
                //
                else if (msg == "PoseIsNowSubscribed")
                {
                    SendText("RequestNextPose");
                }

                //
                // Messages from a module (Analysis)
                //
                else if (msg.StartsWith("Capabilities="))  // Receive capab. from a module
                {
                    // Module is sending their capabilities
                    string[] str_cap = msg.Split('=');
                    string moduleName = str_cap[1];
                    Capability _c = JsonConvert.DeserializeObject<Capability>(str_cap[2]);

                    // Has the module already send their capabilities?
                    Module module = null;
                    foreach (var item in WorldAnalysisConnections.Singleton.modules) if (item.name == moduleName) module = item;
                    if (module != null) module.capabilities.Add(_c);
                }
                else if (msg.StartsWith("NewPose="))
                {
                    string[] str_pose = msg.Split('=');

                    // Send the pose to the client(s)
                    Pose pose = JsonConvert.DeserializeObject<Pose>(str_pose[1]);

                    //WorldAnalysisConnections.Singleton.SendPoseToClients(pose);
                    WorldAnalysisConnections.Singleton.SendPoseToClients(str_pose[1]);

                    // todo: if there are some subscription ask modules for new pose
                    SendText("RequestNextPose");
                }
                else if (msg.StartsWith("Error="))
                {
                    string errorMsg = msg.Split('=')[1];
                    foreach (var item in WorldAnalysisConnections.Singleton.clients)
                    {
                        await SendText(item.websockets.websocket, errorMsg);
                    }
                }

                //
                // Messages from a client (Unity)
                //
                else if (msg.StartsWith("PoseStart")) // send some fake poses to Unity clients
                {
                    await SendDemoPose(websocket);
                }
                else if (msg.StartsWith("CurrentUserPose="))
                {
                    // A client (Unity) is sending the current user posistion

                    // Send new user pose to all modules
                    foreach (var item in WorldAnalysisConnections.Singleton.modules)
                    {
                        await SendText(item.websockets.websocket, msg);
                    }

                    //// Test: send user pose back to clients
                    //string[] str_pose = msg.Split('=');
                    //Pose pose = JsonConvert.DeserializeObject<Pose>(str_pose[1]);
                    //pose.Value.Position[2] += 1f;
                    //WorldAnalysisConnections.Singleton.SendPoseToClients(pose);
                }

                //
                // Messages from modules and clients
                //
                else
                {
                    // Send a response
                    SendText("ARF World Analysis Server: I got this unknown message: " + msg);
                }
            }
        }

        private async Task WebSocketServer_Loop(HttpContext context, WebSocket ws)
        {
            websocket = ws;
            WebSocketControllerInstanceCount++;

            var buffer = new byte[1024 * 4];

            // Read/get the first data block
            WebSocketReceiveResult result = await ws.ReceiveAsync(new ArraySegment<byte>(buffer), CancellationToken.None);
            if (result.MessageType == WebSocketMessageType.Text)
            {
                string getMsg = System.Text.Encoding.UTF8.GetString(buffer, 0, result.Count);
                OnReceiveText(getMsg);
            }

            // Entering the loop
            while (!result.CloseStatus.HasValue)
            {
                // Read/get the next block
                result = await ws.ReceiveAsync(new ArraySegment<byte>(buffer), CancellationToken.None);
                if (result.MessageType == WebSocketMessageType.Text)
                {
                    string getMsg = System.Text.Encoding.UTF8.GetString(buffer, 0, result.Count);
                    OnReceiveText(getMsg);
                }
            }
            await ws.CloseAsync(result.CloseStatus.Value, result.CloseStatusDescription, CancellationToken.None);
            WebSocketControllerInstanceCount--;
        }


        //
        // Echo incoming messages
        //
        private async Task Echo(WebSocketReceiveResult context, WebSocket webSocket)
        {
            var buffer = new byte[1024 * 4];

            // Read/get the first data block
            WebSocketReceiveResult result = await webSocket.ReceiveAsync(new ArraySegment<byte>(buffer), CancellationToken.None);

            // Send/echo the message back to the client
            await SendText(webSocket, "ARF World Analysis Server: I got this (raw) message: ");
            await webSocket.SendAsync(new ArraySegment<byte>(buffer, 0, result.Count), result.MessageType, result.EndOfMessage, CancellationToken.None);
        }
    }
}
#pragma warning restore CS1591 // Fehlendes XML-Kommentar fr ffentlich sichtbaren Typ oder Element


