using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using TMPro;
using UnityEngine;
using UnityEngine.UI;

public class SimulationManager : MonoBehaviour
{
    [Header("Checkpoint (debug)")]
    [SerializeField] private bool startInDebugMode = false;
    [SerializeField] private int _maxCheckpoints;

    [SerializeField] private int _currentCheckpointIndex;
    private GameObject _currentCheckpoint;

    [SerializeField] private TMP_Text _checkpointText;
    private List<GameObject> _debugGOs;

    [Header("AR Object Transform (debug)")]
    [SerializeField] private GameObject _simulationObject;
    [Header("Position")]
    [SerializeField] private TMP_InputField _positionXInputField;
    [SerializeField] private TMP_InputField _positionYInputField;
    [SerializeField] private TMP_InputField _positionZInputField;
    [Header("Rotation")]
    [SerializeField] private TMP_InputField _rotationXInputField;
    [SerializeField] private TMP_InputField _rotationYInputField;
    [SerializeField] private TMP_InputField _rotationZInputField;
    [Header("Scale")]
    [SerializeField] private TMP_InputField _scaleXInputField;
    [SerializeField] private TMP_InputField _scaleYInputField;
    [SerializeField] private TMP_InputField _scaleZInputField;

    [SerializeField] private Button _button;

    private Vector3 _initialPosition;
    private Quaternion _initialRotation;
    private Vector3 _initialScale;

    // Start is called before the first frame update
    void Start()
    {
        _currentCheckpointIndex = 0;

        GetFinalTransformValues();

        // Store the initial transform values
        _initialPosition = _simulationObject.transform.position;
        _initialRotation = _simulationObject.transform.rotation;
        _initialScale = _simulationObject.transform.localScale;

        // Attach button click listener
        _button.onClick.AddListener(OnButtonClick);

        if (!startInDebugMode) ToggleDebugMode();
    }


    public void PreviousCheckpoint()
    {
        if (_currentCheckpointIndex > 0)
        {
            _currentCheckpointIndex--;
            UpdateCheckpointUI();
        }
    }

    public void NextCheckpoint()
    {
        if (_currentCheckpointIndex < _maxCheckpoints - 1)
        {
            _currentCheckpointIndex++;
            UpdateCheckpointUI();
        }
    }

    public void GetFinalTransformValues()
    {
        _positionXInputField.text = _simulationObject.transform.position.x.ToString();
        _positionYInputField.text = _simulationObject.transform.position.y.ToString();
        _positionZInputField.text = _simulationObject.transform.position.z.ToString();

        _rotationXInputField.text = _simulationObject.transform.rotation.eulerAngles.x.ToString();
        _rotationYInputField.text = _simulationObject.transform.rotation.eulerAngles.y.ToString();
        _rotationZInputField.text = _simulationObject.transform.rotation.eulerAngles.z.ToString();

        _scaleXInputField.text = _simulationObject.transform.localScale.x.ToString();
        _scaleYInputField.text = _simulationObject.transform.localScale.y.ToString();
        _scaleZInputField.text = _simulationObject.transform.localScale.z.ToString();
    }

    private void UpdateCheckpointUI()
    {
        _checkpointText.text = "Checkpoint " + (_currentCheckpointIndex + 1);
    }

    private void OnButtonClick()
    {
        OnPositionValueChanged(_positionXInputField.text, _positionYInputField.text, _positionZInputField.text);
        OnRotationValueChanged(_rotationXInputField.text, _rotationYInputField.text, _rotationZInputField.text);
        OnScaleValueChanged(_scaleXInputField.text, _scaleYInputField.text, _scaleZInputField.text);
        GetFinalTransformValues();
    }

    private void OnPositionValueChanged(string xValue, string yValue, string zValue)
    {
        if (float.TryParse(xValue, out float positionX) && float.TryParse(yValue, out float positionY) && float.TryParse(zValue, out float positionZ))
        {
            Debug.Log("Adding position: " + positionX + " " + positionY + " " + positionZ);
            Vector3 newPosition = _simulationObject.transform.localPosition;
            newPosition.x += positionX;
            newPosition.y += positionY;
            newPosition.z += positionZ;
            _simulationObject.transform.position = newPosition;

        }
    }

    private void OnRotationValueChanged(string xValue, string yValue, string zValue)
    {
        if (float.TryParse(xValue, out float rotationX) && float.TryParse(yValue, out float rotationY) && float.TryParse(zValue, out float rotationZ))
        {
            Debug.Log("Adding rotation: " + rotationX + " " + rotationY + " " + rotationZ);
            Quaternion newRotation = _simulationObject.transform.localRotation;
            newRotation.eulerAngles += new Vector3(rotationX, rotationY, rotationZ);
            _simulationObject.transform.rotation = newRotation;

        }
    }

    private void OnScaleValueChanged(string xValue, string yValue, string zValue)
    {
        if (float.TryParse(xValue, out float scaleX) && float.TryParse(yValue, out float scaleY) && float.TryParse(zValue, out float scaleZ))
        {
            Debug.Log("Adding scale: " + scaleX + " " + scaleY + " " + scaleZ);
            Vector3 newScale = _simulationObject.transform.localScale;
            newScale.x += scaleX;
            newScale.y += scaleY;
            newScale.z += scaleZ;
            _simulationObject.transform.localScale = newScale;
        }
    }

    public void ToggleDebugMode()
    {
        if (_debugGOs == null)
        {
            _debugGOs = new List<GameObject>();
            foreach (GameObject go in GameObject.FindGameObjectsWithTag("Debug"))
            {
                _debugGOs.Add(go);
                go.SetActive(!go.activeSelf);
                Debug.Log($"Debug:  {go.name}");
            }
        }
        else
        {
            foreach (GameObject go in _debugGOs)
            {
                go.SetActive(!go.activeSelf);
                Debug.Log($"Debug:  {go.name}");
            }
        }
    }
}
