//
// ARF - Augmented Reality Framework (ETSI ISG ARF)
//
// Copyright 2024 ETSI
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Last change: July 2024
//

using System.Collections;
using UnityEngine;
using UnityEditor;

using ETSI.ARF.OpenAPI;
using ETSI.ARF.OpenAPI.WorldAnalysis;
using ETSI.ARF.WorldAnalysis;
using ETSI.ARF.WorldAnalysis.REST;

[CustomEditor(typeof(WorldAnalysisServer))]
public class WorldStorageServerEditor : Editor
{
    WorldAnalysisServer server;

    private string state = "";
    private string version = "";

    private string test = "";

    private Queue handleResponseQueue = new Queue();

    private ResponseObject<string> pendingTest = null;
    private ResponseObject<string> pendingState = null;
    private ResponseObject<string> pendingVersion = null;

    public void OnEnable()
    {
        server = (WorldAnalysisServer)target;
    }

    public override void OnInspectorGUI()
    {
        serializedObject.Update();

        DrawDefaultInspector();
        EditorGUILayout.Space();

        if (GUILayout.Button("Test server"))
        {
            TestPing();
        }
        EditorGUILayout.LabelField("Test Response", test);

        EditorGUILayout.Space();

        if (GUILayout.Button("Query server"))
        {
            QueryServer();
        }

        EditorGUILayout.LabelField("Server State", state);
        EditorGUILayout.LabelField("OpenAPI Version", version);

        if (handleResponseQueue.Count > 0)
        {
            object o = handleResponseQueue.Dequeue();
            if (o.Equals(pendingTest))
            {
                ResponseObject<string> response = o as ResponseObject<string>;
                Debug.Log($"Get '{response.result}' from server");

                test = response.result;
                pendingTest = null;
                EditorUtility.SetDirty(target);
            }
            else if (o.Equals(pendingState))
            {
                ResponseObject<string> response = o as ResponseObject<string>;
                Debug.Log($"Get '{response.result}' from server");

                state = response.result;
                pendingState = null;
                EditorUtility.SetDirty(target);
            }
            else if (o.Equals(pendingVersion))
            {
                ResponseObject<string> response = o as ResponseObject<string>;
                Debug.Log($"Get '{response.result}' from server");

                version = response.result;
                pendingVersion = null;
                EditorUtility.SetDirty(target);
            }
            else
            {
                Debug.Log("Unsupported response!");
            }
        }
    }

    public override bool RequiresConstantRepaint()
    {
        return handleResponseQueue.Count > 0;
    }

    void OnSceneGUI()
    {
        Debug.Log("OnSceneGUI");
    }

    private void TestPing()
    {
        test = "";
        EditorUtility.SetDirty(target);

        if (server == null)
        {
            Debug.LogError("No server defined!");
            return;
        }

        //string response = AdminRequest.PingSync(server);
        //EditorUtility.DisplayDialog("Test Server", $"Get '{response}' from server", "OK");

        if (pendingTest != null)
        {
            pendingTest.Cancel();
        }

        pendingTest = AdminRequest.PingAsync(server, (response) =>
        {
            handleResponseQueue.Enqueue(response);
            Debug.Log($"Request Time: { response.requestTime.ToLongTimeString() } / Total Time: { response.DeltaTime.TotalMilliseconds }ms\n\n<b>Content:</b>\n{ response.result }");
        });
        
        Debug.Log("Starting request @ time: " + pendingTest.requestTime.ToLongTimeString() + "...");
    }

    private void QueryServer()
    {
        version = "";
        state = "";

        if (pendingState != null)
        {
            pendingState.Cancel();
        }
        
        pendingState = AdminRequest.AdminAsync(server, (response) =>
        {
            handleResponseQueue.Enqueue(response);
        });

        if (pendingVersion != null)
        {
            pendingVersion.Cancel();
        }

        pendingVersion = AdminRequest.VersionAsync(server, (response) =>
        {
            handleResponseQueue.Enqueue(response);
        });
    }
}