using System;
using System.Collections.Generic;
using UnityEngine;

public class WorldAnalysisUnityHelper 
{
    /// <summary>
    /// Convert a float array of length 3 to a Vector3
    /// </summary>
    /// <param name="value">the values to convert</param>
    /// <returns>Converted Unity Vector3</returns>
    /// <exception cref="ArgumentException"></exception>
    public static Vector3 ConvertETSIVector3ToUnity(ETSI.ARF.OpenAPI.WorldAnalysis.Vector3 value)
    {
        if (value.Count == 3) return new Vector3(value[0], value[1], value[2]);
        throw new ArgumentException("The numer of floats in the value parameter must be 3!");
    }

    /// <summary>
    ///  Convert a float array of length 4 to a Quaternion
    /// </summary>
    /// <param name="value">the values to convert</param>
    /// <returns>Converted Unity Quaternion</returns>
    /// <exception cref="ArgumentException"></exception>
    public static Quaternion ConvertETSIARFQuaternionToUnity(ETSI.ARF.OpenAPI.WorldAnalysis.Quaternion value)
    {
        if (value.Count == 4) return new Quaternion(value[0], value[1], value[2], value[3]);
        throw new ArgumentException("The numer of floats in the value parameter must be 4!");
    }

    /// <summary>
    /// Convert a float array of length 16 to a Matrix
    /// </summary>
    /// <param name="matrix">the values to convert</param>
    /// <returns>Converted Unity Matrix</returns>
    public static Matrix4x4 ConvertETSIARFTransform3DToUnity(ETSI.ARF.OpenAPI.WorldAnalysis.Transform3D value)
    {
        if (value.Count == 16)
        {
            Matrix4x4 resul = new Matrix4x4();
            resul[0, 0] = value[0];
            resul[0, 1] = value[1];
            resul[0, 2] = value[2];
            resul[0, 3] = value[3];

            resul[1, 0] = value[4];
            resul[1, 1] = value[5];
            resul[1, 2] = value[6];
            resul[1, 3] = value[7];

            resul[2, 0] = value[8];
            resul[2, 1] = value[9];
            resul[2, 2] = value[10];
            resul[2, 3] = value[11];

            resul[3, 0] = value[12];
            resul[3, 1] = value[13];
            resul[3, 2] = value[14];
            resul[3, 3] = value[15];

            return resul;
        }
        else
        {
            throw new ArgumentException("The numer of floats in the value parameter must be 16!");
        }  
    }

    /// <summary>
    /// Function to extract translation from a Matrix4x4
    /// </summary>
    /// <param name="matrix"></param>
    /// <returns></returns>
    public static Vector3 ExtractTranslationFromMatrix(Matrix4x4 matrix)
    {
        return matrix.GetColumn(3);
    }

    /// <summary>
    /// Function to extract rotation from a Matrix4x4
    /// </summary>
    /// <param name="matrix"></param>
    /// <returns></returns>
    public static Quaternion ExtractRotationFromMatrix(Matrix4x4 matrix)
    {
        // Convert the matrix to a Quaternion
        return Quaternion.LookRotation(matrix.GetColumn(2), matrix.GetColumn(1));
    }

    /// <summary>
    /// Duplicated generated code means that we need conversion between similar types
    /// </summary>
    /// <param name="value">world storage capability</param>
    /// <returns>clone</returns>
    public static ETSI.ARF.OpenAPI.WorldAnalysis.Capability ConvertWorldStorageCapability(ETSI.ARF.OpenAPI.WorldStorage.Capability value)
    {
        ETSI.ARF.OpenAPI.WorldAnalysis.Capability resul = new ETSI.ARF.OpenAPI.WorldAnalysis.Capability();

        resul.TrackableType = (ETSI.ARF.OpenAPI.WorldAnalysis.TrackableType)((int)value.TrackableType);
        resul.EncodingInformation = ConvertWorldStorageEncodingInformation(value.EncodingInformation);
        resul.Framerate = value.Framerate;
        resul.Latency = value.Latency;
        resul.Accuracy = value.Accuracy;
        resul.AdditionalProperties = value.AdditionalProperties; // need copy?

        return resul;
    }

    /// <summary>
    /// Duplicated generated code means that we need conversion between similar types
    /// </summary>
    /// <param name="value">world storage encodinginformation</param>
    /// <returns>clone</returns>
    public static ETSI.ARF.OpenAPI.WorldAnalysis.EncodingInformationStructure ConvertWorldStorageEncodingInformation(ETSI.ARF.OpenAPI.WorldStorage.EncodingInformationStructure value)
    {
        ETSI.ARF.OpenAPI.WorldAnalysis.EncodingInformationStructure resul = new ETSI.ARF.OpenAPI.WorldAnalysis.EncodingInformationStructure();
        resul.DataFormat = (ETSI.ARF.OpenAPI.WorldAnalysis.EncodingInformationStructureDataFormat)((int)value.DataFormat);
        resul.Version = value.Version;
        resul.AdditionalProperties = value.AdditionalProperties; // need copy ?
        return resul;
    }

    /// <summary>
    /// Duplicated generated code means that we need conversion between similar types
    /// </summary>
    /// <param name="value">world storage capability</param>
    /// <returns>clone</returns>
    public static ETSI.ARF.OpenAPI.WorldStorage.Capability ConvertWorldAnalysisCapability(ETSI.ARF.OpenAPI.WorldAnalysis.Capability value)
    {
        ETSI.ARF.OpenAPI.WorldStorage.Capability resul = new ETSI.ARF.OpenAPI.WorldStorage.Capability();
        resul.TrackableType = (ETSI.ARF.OpenAPI.WorldStorage.TrackableType)((int)value.TrackableType);
        resul.EncodingInformation = ConvertWorldAnalysisEncodingInformation(value.EncodingInformation);

        resul.Framerate = value.Framerate;
        resul.Latency = value.Latency;
        resul.Accuracy = value.Accuracy;

        resul.AdditionalProperties = value.AdditionalProperties; // need copy?
        return resul;
    }


    /// <summary>
    /// Duplicated generated code means that we need conversion between similar types
    /// </summary>
    /// <param name="value">world analysis encodinginformation</param>
    /// <returns>clone</returns>
    public static ETSI.ARF.OpenAPI.WorldStorage.EncodingInformationStructure ConvertWorldAnalysisEncodingInformation(ETSI.ARF.OpenAPI.WorldAnalysis.EncodingInformationStructure value)
    {
        ETSI.ARF.OpenAPI.WorldStorage.EncodingInformationStructure resul = new ETSI.ARF.OpenAPI.WorldStorage.EncodingInformationStructure();
        resul.DataFormat = (ETSI.ARF.OpenAPI.WorldStorage.EncodingInformationStructureDataFormat)((int)value.DataFormat);
        resul.Version = value.Version;
        resul.AdditionalProperties = value.AdditionalProperties; // need copy ?
        return resul;
    }

}

namespace ETSI.ARF.OpenAPI.WorldAnalysis
 {
    /// <summary>
    /// Override generated Vector3 constructor
    /// </summary>
    public partial class Vector3 : System.Collections.ObjectModel.Collection<float>
    {
        public Vector3(float x, float y, float z) : base(new List<float> { x, y, z })
        {
        }
    }

    /// <summary>
    /// Override generated Quaternion constructor
    /// </summary>
    public partial class Quaternion : System.Collections.ObjectModel.Collection<float>
    {
        public Quaternion(float x, float y, float z, float w) : base(new List<float> { x, y, z, w})
        {
        }
    }
}