using ETSI.ARF.OpenAPI.WorldAnalysis;
using System;

public interface WorldAnalysisInterface
{
    public static WorldAnalysisInterface Instance;

    #region ResponseCode

    /// <summary>
    /// Ask framerate result
    /// </summary>
    public enum AskFrameRateResult
    {
        OK, // subscription ok
        NOT_SUPPORTED, /// trackable type not supported on that platform
        FRAMERATE_NOT_SUPPORTED, // framerate cannot be reached RQ-AR8-007 The WA function shall provide the pose at the minimum frame rate requested by the SM function, or shall notify the SM function if the requested minimum frame rate cannot be supported
        NONE // unknown
    }

    /// <summary>
    /// Result of the process of a pose estimation
    /// </summary>
    public enum PoseEstimationResult
    {
        OK,
        UNKNOWN_ID,
        NOT_ALLOWED,
        NOT_SUPPORTED,
        FAILURE, // RQ-AR8-005 The WA function shall have the ability to send a notification to the SM function in case of a pose estimation failure. 
        NONE
    }


    /// <summary>
    /// Callback for receiving a pose
    /// </summary>
    /// <param name="estimationResult">last pose estimation result</param>
    /// <param name="pose">last pose</param>
    public delegate void PoseCallback(PoseEstimationResult estimationResult, Pose pose);

    /// <summary>
    /// Result of the process of an operation on the subscription to to the pose of a trackable or anchor
    /// </summary>
    public enum InformationSubscriptionResult
    {
        OK, // subscription operation ok
        UNKNOWN_ID, // trackable or anchor is unknown
        NOT_ALLOWED, // trackable or anchor is not allowed for that user
        NOT_SUPPORTED, /// trackable not supported on that platform
        NONE // unknown
    }

    /// <summary>
    /// Result when getting information about capabilities
    /// </summary>
    public enum CapabilityResult
    {
        OK,
        FAIL,
        UNKNOWN_ID,
        NOT_ALLOWED
    }

    #endregion

    #region Pose
    /// <summary>
    /// Ask the world analysis to estimate the pose of a given trackable type at a given framerate
    /// </summary>
    /// <param name="token">manage access rights RQ-AR8-010 When access control is required by the WA function, the SM function shall provide the WA function with identification information to manage access rights</param>
    /// <param name="type">type of trackable</param>
    /// <param name="encodingInformation"> Encoding information of the trackable type</param>
    /// <param name="minimumFramerate">asked minimum framerate   RQ-AR8-006 The WA function should allow the SM function to specify a minimum frame rate for pose estimation.  </param>
    /// <returns>Result of the set framerate estimation</returns>
    public AskFrameRateResult SetPoseEstimationFramerate(string token, PoseConfigurationTrackableType type, EncodingInformationStructure encodingInformation, int minimumFramerate);

    /// <summary>
    /// Pull the most recent pose of a given trackable or anchor
    /// RQ-AR8-004 The SM function shall have the ability to request the pose for one or several World Anchors or Trackables from the WA function.
    /// </summary>
    /// <param name="token">manage access rights RQ-AR8-010 When access control is required by the WA function, the SM function shall provide the WA function with identification information to manage access rights</param>
    /// <param name="uuid">id of the trackable or the anchor</param>
    /// <param name="mode">trackable/anchor to ar device or opposite RQ-AR8-001 The WA function shall provide the SM function with a pose of either an AR device in relation to a WorldAnchor or a Trackable, or conversely a Trackable or World Anchor in relation to the AR device. </param>
    /// <param name="pose">Last estimated pose</param>
    /// <returns>The last state of the estimation of this pose</returns>
    public PoseEstimationResult GetLastPose(string token, Guid uuid, Mode_WorldAnalysis mode, out Pose pose);

    /// <summary>
    /// Pull the most recent pose of a multiple trackables and anchors
    /// RQ-AR8-004 The SM function shall have the ability to request the pose for one or several World Anchors or Trackables from the WA function.
    /// </summary>
    /// <param name="token">manage access rights RQ-AR8-010 When access control is required by the WA function, the SM function shall provide the WA function with identification information to manage access rights</param>
    /// <param name="uuid">list of ids of trackables and anchors</param>
    /// <param name="mode">list of trackable/anchor to ar device or opposite RQ-AR8-001 The WA function shall provide the SM function with a pose of either an AR device in relation to a WorldAnchor or a Trackable, or conversely a Trackable or World Anchor in relation to the AR device. </param>
    /// <param name="poses">Last estimated pose</param>
    /// <returns>The last state of the estimation of this pose</returns>
    public PoseEstimationResult[] GetLastPoses(string token, Guid[] uuids, Mode_WorldAnalysis [] modes, out Pose [] poses);

    /// <summary>
    /// Subscribe to receive the pose data of a given trackable or anchor
    /// RQ-AR8-004 The SM function shall have the ability to request the pose for one or several World Anchors or Trackables from the WA function.
    /// </summary>
    /// <param name="token">manage access rights RQ-AR8-010 When access control is required by the WA function, the SM function shall provide the WA function with identification information to manage access rights</param>
    /// <param name="uuid">unique identifier of the trackable or anchors</param>
    /// <param name="mode">trackable/anchor to ar device or opposite RQ-AR8-001 The WA function shall provide the SM function with a pose of either an AR device in relation to a WorldAnchor or a Trackable, or conversely a Trackable or World Anchor in relation to the AR device. </param>
    /// <param name="callback">function that is called with the pose</param>
    /// <param name="validity">Subscription validity delay in millisecond</param>
    /// <param name="subscriptionUUID "> id of the callback</param>
    /// <returns>Result of the subscription</returns>
    public InformationSubscriptionResult SubscribeToPose(string token, Guid uuid, Mode_WorldAnalysis mode, PoseCallback callback, ref int validity, out Guid subscriptionUUID);

    /// <summary>
    /// Subscribe to receive the pose data of a multiple trackables and anchors
    /// RQ-AR8-004 The SM function shall have the ability to request the pose for one or several World Anchors or Trackables from the WA function.
    /// </summary>
    /// <param name="token">manage access rights RQ-AR8-010 When access control is required by the WA function, the SM function shall provide the WA function with identification information to manage access rights</param>
    /// <param name="uuids">list of unique identifiers of trackable and anchors</param>
    /// <param name="modes">list of trackable/anchor to ar device or opposite RQ-AR8-001 The WA function shall provide the SM function with a pose of either an AR device in relation to a WorldAnchor or a Trackable, or conversely a Trackable or World Anchor in relation to the AR device.</param>
    /// <param name="callback">function that is called with the pose</param>
    /// <param name="validity">Subscription validity delay in millisecond</param>
    /// <param name="subscriptionUUIDs"> id of the callback</param>
    /// <returns>List of subscription result</returns>
    public InformationSubscriptionResult[] SubscribeToPoses(string token, Guid[] uuids, Mode_WorldAnalysis [] modes, PoseCallback callback, ref int validity, out Guid[] subscriptionUUIDs);

    /// <summary>
    /// Get the subscription and the associated callback for collecting a given pose
    /// </summary>
    /// <param name="token">manage access rights</param>
    /// <param name="subscriptionUUID">Subscription UUID to retrieve</param>
    /// <param name="callback">Pose callback</param>
    /// <param name="target">UUID of the Trackable or Anchor associated with the subscription</param>
    /// <param name="mode">Mode representing the context of the relocalization information (AR device to WorldAnchor/Trackable or WorldAnchor/Trackable to AR device)</param>
    /// <param name="validity">Remaining Subscription validity delay in millisecond</param>
    /// <returns>The subscription exists or not</returns>
    public InformationSubscriptionResult GetSubsription(string token, Guid subscriptionUUID, out PoseCallback callback, out Guid target, out Mode_WorldAnalysis mode, out int validity);

    /// <summary>
    /// Update an existing subscription
    /// </summary>
    /// <param name="token">manage access rights</param>
    /// <param name="subscriptionUUID">Subscription UUID to update</param>
    /// <param name="mode">Mode representing the context of the relocalization information (AR device to WorldAnchor/Trackable or WorldAnchor/Trackable to AR device)</param>
    /// <param name="validity">new validity of the subscription</param>
    /// <param name="callback">new callback to use</param>
    /// <returns>The subscription has been updated or not</returns>
    public InformationSubscriptionResult UpdateSubscription(string token, Guid subscriptionUUID, Mode_WorldAnalysis mode, int validity, PoseCallback callback);

    /// <summary>
    /// Unsubscribe to a pose
    /// </summary>
    /// <param name="subscriptionUUID">id of the subscription</param>
    /// /// <returns>The unsubscription has been performed or not</returns>
    public InformationSubscriptionResult UnSubscribeToPose(Guid subscriptionUUID);

    #endregion

    #region Capabilities

    /// <summary>
    /// Get the capabilities of the world analysis 
    /// RQ-AR8-009 The WA function shall provide the SM function with information about its capabilities (e.g. frame rate, latency, accuracy or Trackable types supported for the pose estimation).
    /// </summary>
    /// <param name="token">manage access rights RQ-AR8-010 When access control is required by the WA function, the SM function shall provide the WA function with identification information to manage access rights</param>
    /// <param name="capabilities">for the list of supported capabilites and associated metadata. A capability not included in the list is not supported</param>
    /// <returns>Capabilities have correctly identified or not</returns>
    public CapabilityResult GetCapabilities(string token, out Capability[] capabilities);

    /// <summary>
    /// For a given trackable or anchor, get its support information  
    /// RQ-AR8-009 The WA function shall provide the SM function with information about its capabilities (e.g. frame rate, latency, accuracy or Trackable types supported for the pose estimation).
    /// </summary>
    /// <param name="token">manage access rights RQ-AR8-010 When access control is required by the WA function, the SM function shall provide the WA function with identification information to manage access rights</param>
    /// <param name="uuid">id of the trackable or anchor</param>
    /// <param name="isSupported">trackable or anchor can be estimated on current system</param>
    /// <param name="type">the specified id corresponds to an anchor or trackable</param>
    /// <param name="capability">list metadata associated with the pose estimation of this element</param>
    /// <returns>Capabilities have correctly identified or not</returns>
    public CapabilityResult GetCapability(string token, Guid uuid, out bool isSupported, out TypeWorldStorage type, out Capability [] capability);

    #endregion
}