#if UNITY_ANDROID && ETSIARF_ARCORE_EXTENSIONS
using System;
using System.Collections;
using System.Collections.Generic;
using Google.XR.ARCoreExtensions;
using Unity.XR.CoreUtils;
using UnityEngine;
using UnityEngine.XR.ARFoundation;
using UnityEngine.XR.ARSubsystems;
using static WorldAnalysisARFoundationModule;

public class WorldAnalysisARFoundationModuleARCoreAnchor : WorldAnalysisARFoundationModule 
{
    /// <summary>
    ///  Anchor manager
    /// </summary>
    private ARAnchorManager m_anchorManager;
    /// <summary>
    /// List of cloud anchors with tracking infos
    /// </summary>
    private Dictionary<string, TrackableInfo> m_trackedCloudAnchors = new Dictionary<string, TrackableInfo>();
    /// <summary>
    /// Correspondance between local trackable and etsi uuid
    /// </summary>
    private Dictionary<string, string> m_localIdToEtsiId = new Dictionary<string, string>();

    /// <summary>
    /// Initialize ARCore cloud anchors tracking module
    /// </summary>
    public void Initialize()
    {
        XROrigin origin = UnityEngine.Object.FindAnyObjectByType<XROrigin>();
        m_anchorManager = origin.gameObject.GetComponent<ARAnchorManager>();
        if (m_anchorManager == null )
        {
            m_anchorManager = origin.gameObject.AddComponent<ARAnchorManager>();
            GameObject anchorPrefab = (GameObject)Resources.Load("ARFAnchorTrackingPrefab");
            m_anchorManager.anchorPrefab = anchorPrefab;
        }
        m_anchorManager.anchorsChanged += OnTrackedCloudAnchorChanged;
    }

    /// <summary>
    /// Get the pose of the trackable from its uuid
    /// </summary>
    /// <param name="uuid">id of the trackable</param>
    /// <returns>null or trackableInfo with last updated values</returns>
    public TrackableInfo GetPoseTrackable(Guid uuid)
    {
        if (m_trackedCloudAnchors.ContainsKey(uuid.ToString()))
        {
            return m_trackedCloudAnchors[uuid.ToString()];
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// Need to be a map
    /// </summary>
    /// <param name="trackable"></param>
    /// <returns>map or not (does not check is solved)</returns>
    public bool AddTrackable(ETSI.ARF.OpenAPI.WorldStorage.Trackable trackable)
    {
        /// Here : we don't check if the trackable is allready added, AddImageToLibrary does it
        if (trackable.TrackableType != ETSI.ARF.OpenAPI.WorldStorage.TrackableType.MAP)
        {
            return false;
        }
        AddNewARCoreCloudAnchor(trackable); // here we don't check if the cloud anchor is correctly resolved, this could be imrpoved but would require this method to be async (change of api)
        return true;
    }

    /// <summary>
    /// Initialize capability object with the features of the arcore cloud anchor module
    /// </summary>
    /// <returns>a Capability object</returns>
    public ETSI.ARF.OpenAPI.WorldAnalysis.Capability GetSupportedCapability()
    {
        ETSI.ARF.OpenAPI.WorldAnalysis.Capability capabilityARCoreAnchor = new ETSI.ARF.OpenAPI.WorldAnalysis.Capability();
        capabilityARCoreAnchor.TrackableType = ETSI.ARF.OpenAPI.WorldAnalysis.TrackableType.MAP;
        ETSI.ARF.OpenAPI.WorldAnalysis.EncodingInformationStructure encodingInformation = new ETSI.ARF.OpenAPI.WorldAnalysis.EncodingInformationStructure();
        encodingInformation.DataFormat = ETSI.ARF.OpenAPI.WorldAnalysis.EncodingInformationStructureDataFormat.ARCORE;
        encodingInformation.Version = "1.01";
        capabilityARCoreAnchor.EncodingInformation = encodingInformation;
        capabilityARCoreAnchor.Framerate = 30; // Not particularly specified on ARKit and ARCore
        capabilityARCoreAnchor.Latency = 0; // Not particularly specified on ARKit and ARCore
        capabilityARCoreAnchor.Accuracy = 1; // Not particularly specified on ARKit and ARCore
        return capabilityARCoreAnchor;
    }

    /// <summary>
    /// Update arcore cloud anchors informations
    /// </summary>
    /// <param name="eventArgs"></param>
    private void OnTrackedCloudAnchorChanged(ARAnchorsChangedEventArgs eventArgs)
    {
        foreach (var trackedCloudAnchor in eventArgs.updated)
        {
            Vector3 position = trackedCloudAnchor.transform.position;
            Quaternion rotation = trackedCloudAnchor.transform.rotation;


            TrackableInfo info = new TrackableInfo();
            info.name = trackedCloudAnchor.name;
            string localId = trackedCloudAnchor.trackableId.subId1.ToString("X16");// there must be a better way : does it work every time?
            if (m_localIdToEtsiId.ContainsKey(localId))
            {
                info.name = m_localIdToEtsiId[localId];
            }
            else
            {
                Debug.Log("ARCore Cloud Anchor No correspondance for Local Anchor " + localId);
                continue; 
            }

            if (trackedCloudAnchor.trackingState == TrackingState.Tracking)
            {
                info.state = ETSI.ARF.OpenAPI.WorldAnalysis.PoseEstimationState.OK;
                info.confidence = 100;
            }
            else if (trackedCloudAnchor.trackingState == TrackingState.Limited)
            {
                info.state = ETSI.ARF.OpenAPI.WorldAnalysis.PoseEstimationState.OK;
                info.confidence = 50;
            }
            else
            {
                info.state = ETSI.ARF.OpenAPI.WorldAnalysis.PoseEstimationState.FAILURE; //ADD NOT_TRACKED ?
                info.confidence = 0;
            }
            info.timeStamp = unchecked((int)DateTimeOffset.UtcNow.ToUnixTimeMilliseconds());
            info.position = position;
            info.rotation = rotation;
            info.trackableType = ETSI.ARF.OpenAPI.WorldAnalysis.TrackableType.MAP;
            m_trackedCloudAnchors[info.name] = info;
        }
    }

    /// <summary>
    /// Create a new ARCore Cloud anchor using ARCore API id 
    /// </summary>
    /// <param name="trackable"></param>
    private async void AddNewARCoreCloudAnchor(ETSI.ARF.OpenAPI.WorldStorage.Trackable trackable)
    {
        while (ARSession.state == ARSessionState.CheckingAvailability || ARSession.state == ARSessionState.None ||ARSession.state == ARSessionState.SessionInitializing)
        {
            await System.Threading.Tasks.Task.Delay(100);
        }
        string googleCloudID = trackable.Name;
        ResolveCloudAnchor(googleCloudID, trackable.UUID.ToString());
    }

    /// <summary>
    /// Resolves Cloud Anchors using a given id
    /// </summary>
    /// <param name="cloudId"></param>
    public void ResolveCloudAnchor(string cloudId, string etsiId)
    {
        var promise = m_anchorManager.ResolveCloudAnchorAsync(cloudId);
        if (promise.State == PromiseState.Done)
        {
            Debug.Log("Failed to resolve Cloud Anchor " + cloudId);
        }
        else
        {
            var coroutine = ResolveAnchor(promise, cloudId, etsiId);
            WorldAnalysisARFoundationCoroutineHelper.Instance.StartACoroutine(coroutine);
        }
    }

    /// <summary>
    /// Resolves cloud anchor promise
    /// </summary>
    /// <param name="promise"></param>
    /// <returns></returns>
    private IEnumerator ResolveAnchor(ResolveCloudAnchorPromise promise, string cloudId, string etsiId)
    {
        yield return promise;
        var result = promise.Result;
        if (result.CloudAnchorState == CloudAnchorState.Success)
        {
            Debug.Log("ARCloud Anchor Resolve Sucess" +cloudId);
            m_localIdToEtsiId.Add(result.Anchor.trackableId.subId2.ToString("X16"), etsiId); // should be a better way: not sure about that but subId2 of the ARCloudAnchor seems to correspond to subId1 of local anchor that is updated by Anchor Manager
        }
        else
        {
            Debug.Log("ARCloud Anchor Resolve Failed" + result.CloudAnchorState +  "    " +cloudId);
        }
    }
}

#endif