#if UNITY_ANDROID
using System;
using System.Collections;
using System.Collections.Generic;
using Google.XR.ARCoreExtensions;
using Unity.XR.CoreUtils;
using UnityEngine;
using UnityEngine.XR.ARFoundation;
using UnityEngine.XR.ARSubsystems;
using static WorldAnalysisARFoundationModule;

public class WorldAnalysisARFoundationModuleARCoreAnchor : WorldAnalysisARFoundationModule 
{
    /// <summary>
    /// 
    /// </summary>
    private ARAnchorManager m_anchorManager;
    /// <summary>
    /// IDs of all the cloud anchors that have been resolved 
    /// </summary>
    private List<string> m_cloudAnchorAdded;//useful ?
    /// <summary>
    /// List of cloud anchors with tracking infos
    /// </summary>
    private Dictionary<string, TrackableInfo> m_trackedCloudAnchors = new Dictionary<string, TrackableInfo>();
    /// <summary>
    /// 
    /// </summary>
    private List<ResolveCloudAnchorPromise> _resolvePromises = new List<ResolveCloudAnchorPromise>();
    /// <summary>
    /// 
    /// </summary>
    private List<ResolveCloudAnchorResult> _resolveResults = new List<ResolveCloudAnchorResult>();
    /// <summary>
    /// Correspondance between google cloud id and local trackable id
    /// </summary>
    private Dictionary<string, string> m_localIdToCloudId = new Dictionary<string, string>();

    /// <summary>
    /// ETSI ID
    /// </summary>
    private Dictionary<string, string> m_CloudIdToETSIId = new Dictionary<string, string>();

    /// <summary>
    /// 
    /// </summary>
    private GameObject m_anchorPrefab;

    /// <summary>
    /// Initialize ARCore cloud anchors tracking module
    /// </summary>
    public void Initialize()
    {
        XROrigin origin = UnityEngine.Object.FindAnyObjectByType<XROrigin>();
        m_anchorManager = origin.gameObject.AddComponent<ARAnchorManager>();
        m_cloudAnchorAdded = new List<string>();
        m_anchorPrefab = (GameObject)Resources.Load("ARFAnchorTrackingPrefab");
        m_anchorManager.anchorPrefab = m_anchorPrefab;
        m_anchorManager.anchorsChanged += OnTrackedCloudAnchorChanged;
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="uuid"></param>
    /// <returns></returns>
    public TrackableInfo GetPoseTrackable(Guid uuid)
    {
        if (m_trackedCloudAnchors.ContainsKey(uuid.ToString()))
        {
            return m_trackedCloudAnchors[uuid.ToString()];
        }
        else
        {
            return null;
        }
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="trackable"></param>
    /// <returns></returns>
    public bool AddTrackable(ETSI.ARF.OpenAPI.WorldStorage.Trackable trackable)
    {
        /// Here : we don't check if the trackable is allready added, AddImageToLibrary does it
        if (trackable.TrackableType != ETSI.ARF.OpenAPI.WorldStorage.TrackableType.MAP)
        {
            return false;
        }
        AddNewARCoreCloudAnchor(trackable);
        return true;
    }

    /// <summary>
    /// 
    /// </summary>
    /// <returns></returns>
    public ETSI.ARF.OpenAPI.WorldAnalysis.Capability GetSupportedCapability()
    {
        ETSI.ARF.OpenAPI.WorldAnalysis.Capability capabilityARCoreAnchor = new ETSI.ARF.OpenAPI.WorldAnalysis.Capability();
        capabilityARCoreAnchor.TrackableType = ETSI.ARF.OpenAPI.WorldAnalysis.TrackableType.MAP;
        ETSI.ARF.OpenAPI.WorldAnalysis.EncodingInformationStructure encodingInformation = new ETSI.ARF.OpenAPI.WorldAnalysis.EncodingInformationStructure();
        encodingInformation.DataFormat = ETSI.ARF.OpenAPI.WorldAnalysis.EncodingInformationStructureDataFormat.ARCORE;
        encodingInformation.Version = "1.01";
        capabilityARCoreAnchor.EncodingInformation = encodingInformation;
        capabilityARCoreAnchor.Framerate = 30; // Not particularly specified on ARKit and ARCore
        capabilityARCoreAnchor.Latency = 0; // Not particularly specified on ARKit and ARCore
        capabilityARCoreAnchor.Accuracy = 1; // Not particularly specified on ARKit and ARCore
        return capabilityARCoreAnchor;
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="eventArgs"></param>
    private void OnTrackedCloudAnchorChanged(ARAnchorsChangedEventArgs eventArgs)
    {
        foreach (var trackedCloudAnchor in eventArgs.updated)
        {
            Vector3 position = trackedCloudAnchor.transform.position;
            Quaternion rotation = trackedCloudAnchor.transform.rotation;


            TrackableInfo info = new TrackableInfo();
            info.name = trackedCloudAnchor.name;
            string localId = trackedCloudAnchor.trackableId.subId1.ToString("X16");// there must be a better way : does it work every time?
            if (m_localIdToCloudId.ContainsKey(localId))
            {
                info.name =  m_CloudIdToETSIId[m_localIdToCloudId[localId]];
            }
            else
            {
                Debug.Log("ARCore Cloud Anchor No correspondance for Local Anchor " + localId);
                continue; 
            }

            if (trackedCloudAnchor.trackingState == TrackingState.Tracking)
            {
                info.state = ETSI.ARF.OpenAPI.WorldAnalysis.PoseEstimationState.OK;
                info.confidence = 100;
            }
            else if (trackedCloudAnchor.trackingState == TrackingState.Limited)
            {
                info.state = ETSI.ARF.OpenAPI.WorldAnalysis.PoseEstimationState.OK;
                info.confidence = 50;
            }
            else
            {
                info.state = ETSI.ARF.OpenAPI.WorldAnalysis.PoseEstimationState.FAILURE; //ADD NOT_TRACKED ?
                info.confidence = 0;
            }
            info.timeStamp = unchecked((int)DateTimeOffset.UtcNow.ToUnixTimeMilliseconds());
            info.position = position;
            info.rotation = rotation;
            info.trackableType = ETSI.ARF.OpenAPI.WorldAnalysis.TrackableType.IMAGE_MARKER;
            m_trackedCloudAnchors[info.name] = info;
        }
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="trackable"></param>
    private async void AddNewARCoreCloudAnchor(ETSI.ARF.OpenAPI.WorldStorage.Trackable trackable )
    {
        while (ARSession.state == ARSessionState.CheckingAvailability || ARSession.state == ARSessionState.None ||ARSession.state == ARSessionState.SessionInitializing)
        {
            await System.Threading.Tasks.Task.Delay(100);
        }
        string googleCloudID = trackable.Name;
        m_CloudIdToETSIId.Add(googleCloudID, trackable.UUID.ToString());
        ResolveCloudAnchor(googleCloudID);
    }

    /// <summary>
    /// Resolves Cloud Anchors using a given id
    /// </summary>
    /// <param name="cloudId"></param>
    public void ResolveCloudAnchor(string cloudId)
    {
        var promise = m_anchorManager.ResolveCloudAnchorAsync(cloudId);
        if (promise.State == PromiseState.Done)
        {
            Debug.LogFormat("Failed to resolve Cloud Anchor " + cloudId);
        }
        else
        {
            _resolvePromises.Add(promise);
            var coroutine = ResolveAnchor(promise, cloudId);
            WorldAnalysisARFoundationCoroutineHelper.Instance.StartACoroutine(coroutine);
        }
    }

    /// <summary>
    /// 
    /// </summary>
    /// <param name="promise"></param>
    /// <returns></returns>
    private IEnumerator ResolveAnchor(ResolveCloudAnchorPromise promise, string cloudId)
    {
        yield return promise;
        var result = promise.Result;
        _resolvePromises.Remove(promise);
        _resolveResults.Add(result);

        if (result.CloudAnchorState == CloudAnchorState.Success)
        {
            Debug.Log("ARCloud Anchor Resolve Sucess" +cloudId);
            m_localIdToCloudId.Add(result.Anchor.trackableId.subId2.ToString("X16"), cloudId); // should be a better way: not sure about that but subId2 of the ARCloudAnchor seems to correspond to subId1 of local anchor that is updated by Anchor Manager
        }
        else
        {
            Debug.Log("ARCloud Anchor Resolve Failed" + result.CloudAnchorState +  "    " +cloudId);
        }
    }
}

#endif